/* 
     File : GatewayMetricVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : An extension to the basic metric visitor which considers
            gateways as the receivers.
             
*/

package mwalk.visitor;


import java.util.Date;
import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.io.PrintStream;
import java.io.FileOutputStream;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Config;
import mwalk.util.MetricCount;
import mwalk.util.VisitException;


/**
 * An extension to the basic metric visitor which considers
 * gateways as the receivers.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class GatewayMetricVisitor extends BasicMetricVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "GatewayMetricVisitor.";
    /** Default metric file name if none supplied */
    private static final String DEF_FILENAME = "tree-metric.gateway";

    protected PrintStream gps;
    protected String gfilename = null;
    /** Default gateway file name if none supplied */
    protected String def_gfilename = "tree-gateway";

    protected int gateways = 0;
    protected int receivers = 0;


    /**
     * Default constructor.
     */
    public GatewayMetricVisitor() {

	super( DEF_FILENAME );
    }

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     * @param <code>String</code> new default gateway filename
     */
    public GatewayMetricVisitor( String metricFile, String gatewayFile ) {

	super( metricFile );

	def_filename = gatewayFile;
    }

    /**
     * Constructor.
     *
     * @param <code>long</code> number of periods to cover session
     */
    public GatewayMetricVisitor( long periods ) {

	super( DEF_FILENAME, periods );
    }

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     * @param <code>String</code> new default gateway filename
     * @param <code>long</code> number of periods to cover session
     */
    public GatewayMetricVisitor( String metricFile, String gatewayFile, long periods ) {

	super( metricFile, periods );

	def_gfilename = gatewayFile;
    }


    /**
     * Initialize the visitor state.
     *
     * @param <code>Tree</code> current tree instance
     */
    public boolean init( Tree tree ) {

	// get filename from config
	gfilename = cfgString( CFG_BASE + "filename" );
	if( gfilename == null )
	    gfilename = def_gfilename;

	super.init( tree );

	String visitor = getClass().getName();
	try {
	    gps = new PrintStream( new FileOutputStream( gfilename ) );
	    gps.println( "# " + visitor + " output data - " + (new Date()).toString() );
	    gps.println( "# simulation start:" + tree.getStart() + " stop:" + tree.getStop() );
	    gps.println( "#" );
	    gps.println( "# time\tGateways\tReceivers\tAvg." );

	} catch( Exception e ) {
	    Config.verbose( visitor + ".init(): error preparing gateway file" );
	    return( false );
	}

	return( true );
    }

    /**
     * Reset the visitor for a new pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     */
    public void reset( Tree tree ) {

	super.reset( tree );

	gateways = receivers = 0;
    }


    protected MetricCount doVisit( Tree tree, TreeNode node, Vector list ) {
	
	MetricCount count = new MetricCount();
	int gateway = 0;
	int klinks = 0;
	
	// handle counts from each child link
	for( Enumeration kids = list.elements(); kids.hasMoreElements(); ) 
	    try {
		MetricCount kcount = (MetricCount)kids.nextElement();
		if( kcount.receivers > 0 )
		    if( kcount.fromReceiver )
			gateway++;
		    else {
			count.add( kcount );
			klinks++;
		    }
		
	    } catch( NoSuchElementException nse ) {}
	
	
	// add ourselves into the pot
	addLinks( count, klinks );

	// if we have directly connected receivers, add ourselves as a representative receiver
	if( gateway > 0 ) {
	    count.receivers++;
	    // keep stats on gateway fan-out
	    gateways++;
	    receivers += gateway;
	}
	
	return( count );
    }


    protected void process( Tree tree ) throws VisitException {

	super.process( tree );

	try {
	    // print gateway counts	    
	    double avg = (gateways > 0) ? (double)receivers / (double)gateways : 0.0;

	    // build a buffer with each field separated by tabs
	    StringBuffer buf = new StringBuffer( 128 );
	    char tab = '\t';
	    buf.append( time ).append( tab ).append( gateways ).append( tab );
	    buf.append( receivers ).append( tab ).append( avg );

	    // convert buffer to one string
	    String line = buf.toString();
	    Config.verbose( this, line );
	    gps.println( line );

	} catch( Exception e ) {
	    throw new VisitException( "error printing gateway data at time " + time, e );
	}
    }
}
