/* 
     File : MetricVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : Abstract base class defining the looping behavior of
            most metric visitors.
             
*/

package mwalk.visitor;


import java.util.Vector;

import mwalk.core.*;
import mwalk.util.MetricCount;
import mwalk.util.VisitException;


/**
 * Abstract base class defining the looping behavior of most metric visitors.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public abstract class MetricVisitor extends PeriodicVisitor implements DownVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "MetricVisitor.";
    /** Default file name if none supplied */
    private static final String DEF_FILENAME = "tree-metric";


    /**
     * Default constructor.
     */
    public MetricVisitor() {

	super( DEF_FILENAME );
    }

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     */
    public MetricVisitor( String filename ) {

	super( filename );
    }

    /**
     * Constructor.
     *
     * @param <code>long</code> number of periods to cover session
     */
    public MetricVisitor( long periods ) {

	super( DEF_FILENAME, periods );
    }

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     * @param <code>long</code> number of periods to cover session
     */
    public MetricVisitor( String filename, long periods ) {

	super( filename, periods );
    }


    /**
     * Whether to visit after visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit after
     */   
    public boolean postfix() {

	return( true );
    }

    /**
     * Visit the current node prior to visiting children.
     * This method is called if prefix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node )  throws VisitException {

	// mark node as seen and visited
	markVisited( node );
	// just return a simple metric count for a single receiver (active or not)
	return( new MetricCount( eval.isActive( (Receiver)node, time ) ) );
    }

    /**
     * Visit the current node after visiting children.
     * This method is called if postfix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Vector</code> list of return values passed back along the path
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node, Vector list ) throws VisitException {

	MetricCount count = null;

	// check if this node has been visited yet
	if( ! markVisited( node ) ) {

	    // handle the visit (make it easier to sub-class)
	    count = doVisit( tree, node, list );

	    // save count in source
	    if( node instanceof Source )
		node.data.put( "metric", count );
	}

	// return total count for this sub-tree
	return( count );
    }

    /**
     * Actually perform visit in derived class.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Vector</code> list of return values passed back along the path
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    protected abstract MetricCount doVisit( Tree tree, TreeNode node, Vector list ) throws VisitException;
}
