/* 
     File : PathlessVisitor.java
   Author : Robert Chalmers

 Original : August 17, 2001
  Revised : 

  Content : A visitor used to remove paths that do not lead back to the source.

  $Id: PathlessVisitor.java,v 1.1 2001/08/22 17:44:36 robertc Exp $
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.core.Link;
import mwalk.core.Config;
import mwalk.util.VisitException;


/**
 * A visitor used to remove paths that do not lead back to the source.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class PathlessVisitor extends AbstractVisitor implements UpVisitor {

    /** List of receivers to remove from the tree */
    protected Vector receivers = new Vector( 10, 10 );

    /** Count of nodes removed */
    protected int nodes = 0;
    /** Count of edges removed */
    protected int edges = 0;


    /**
     * Default constructor.
     */
    public PathlessVisitor() {}


    /**
     * Return the number of pathless receivers.
     *
     * @return <code>int</code> number of pathless receivers.
     */
    public int countReceivers() {
	
	return( receivers.size() );
    }

    /**
     * Return the number of pathless nodes in the graph.
     * This count does not include receivers being removed.
     *
     * @return <code>int</code> number of pathless nodes
     */
    public int countNodes() {

	return( nodes );
    }

    /**
     * Return the number of pathless edges in the graph.
     *
     * @return <code>int</code> number of pathless edges
     */
    public int countEdges() {

	return( edges );
    }

    /**
     * Remove receivers that were marked as pathless.
     *
     * @param <code>Tree</code> current tree
     * @param <code>int</code> number of receivers removed
     */
    public int removeReceivers( Tree tree ) {

	for( Enumeration recvs = receivers.elements(); recvs.hasMoreElements(); )
	    try {
		// delete each receiver
		tree.delNode( (TreeNode)recvs.nextElement() );
		
	    } catch( NoSuchElementException nse ) {
		Config.verbose( "Ran out of receivers in PathlessVisitor.remove()" );
	    }
	
	return( receivers.size() );
    }


    /**
     * Whether to visit after visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit after
     */   
    public boolean postfix() {

	return( true );
    }

    /**
     * Reset the visitor for a new pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     */
    public void reset( Tree tree ) {
	
	super.reset( tree );

	// create list of receivers and clear node and edge counts
	receivers = new Vector( 10, 10 );
	nodes = edges = 0;
    }


    /**
     * Visit the current node after visiting parent.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Object</code> list of return values passed back along the path
     * @return <code>Vector</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitUp( Tree tree, TreeNode node, Vector list ) throws VisitException {

	// walk through list of pathless parents 
	for( Enumeration parents = list.elements(); parents.hasMoreElements(); )
	    try {
		// get a reference to parent node
		TreeNode p = (TreeNode)parents.nextElement();

		// increment counts
		nodes++;
		edges += p.children();		
		// delete parent node
		tree.delNode( p );

	    } catch( NoSuchElementException nse ) {
		Config.verbose( "Ran out of parents in PathlessVisitor.visitUp()" );
	    }

	// figure out whether we are now pathless
	if( node.parents() == 0 ) {
	    // if we are the receiver, then add ourselves to the list to be removed
	    if( node instanceof Receiver )
		receivers.add( node );
	    // otherwise, return ourself so our child can kill us off
	    else
		return( node );
	}

	return( null );
    }
}
