/* 
     File : PeriodicVisitor.java
   Author : Robert Chalmers

 Original : May 09, 2002
  Revised : 

  Content : Abstract base class defining the looping behavior of visitors
            that walk the tree based on periodic intervals of the session time.
*/

package mwalk.visitor;


import java.util.Vector;

import mwalk.core.*;
import mwalk.util.VisitException;


/**
 * Abstract base class defining the looping behavior of visitors that walk 
 * the tree based on periodic intervals of the session time.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public abstract class PeriodicVisitor extends PrintVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "PeriodicVisitor.";
    /** Default file name if none supplied */
    private static final String DEF_FILENAME = "tree-periodic";

    protected long time = 0;
    protected double timeval = 0;
    protected double offset = 0;
    protected long periods = 0;

    /**
     * Default constructor.
     */
    public PeriodicVisitor() {

	super( DEF_FILENAME );
    }

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     */
    public PeriodicVisitor( String filename ) {

	super( filename );
    }

    /**
     * Constructor.
     *
     * @param <code>long</code> number of periods to cover session
     */
    public PeriodicVisitor( long periods ) {

	super( DEF_FILENAME );

	this.periods = periods;
    }

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     * @param <code>long</code> number of periods to cover session
     */
    public PeriodicVisitor( String filename, long periods ) {

	super( filename );

	this.periods = periods;
    }


    /**
     * Get the current session time.
     *
     * @return <code>long</code> the current session time
     */
    public long getTime() {

	return( time );
    }

    /**
     * Increment the current session time by the period interval.
     *
     * @return <code>long</code> the current session time
     */
    public long incTime() {

	timeval += offset;
	return( time = (long)timeval );
    }

    /**
     * Set the current session time.
     *
     * @param <code>double</code> the current session time
     */
    public void setTime( double tv ) {

	timeval = tv;
	time = (long)tv;
    }

    /**
     * Set the starting time based on the start of the session.
     *
     * @param <code>Tree</code> current tree instance
     */
    public void startTime( Tree tree ) {

	setTime( (double)tree.getStart() );
    }


    /**
     * Initialize the visitor state.
     *
     * @param <code>Tree</code> current tree instance
     */
    public boolean init( Tree tree ) {

	// get period from config file if not already set
	if( periods == 0 ) {
	    periods = cfgInt( CFG_BASE + "periods" );
	    if( periods <= 0 )
		periods = 1;
	}

	// get the start time for the simulation
	// compute the incremental offset to use
	startTime( tree );
	if( periods > 1 )
	    // set offset to include enough periods to hit start and stop times
	    offset = (double)(tree.getStop() - tree.getStart()) / (double)(periods - 1);
	else
	    // set offset to push beyond stop time
	    offset = (double)(tree.getStop() - tree.getStart() + 1);

	Config.verbose( this, "\nWalking tree at time " + time );

	return( super.init( tree ) );
    }

    /**
     * Check whether the visitor wants another pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     * @return <code>boolean</code> whether to walk again
     */
    public boolean again( Tree tree ) {

	try {
	    // allow derived classes to do something meaningful
	    process( tree );

	} catch( VisitException ve ) {
	    Config.verbose( ve.getMessage() );
	    return( false );
	}

	// reset for the next pass (if more time)
	reset( tree );
	if( incTime() <= tree.getStop() ) {
	    Config.verbose( this, "\nWalking tree at time " + time );
	    return( true );
	} else {
	    startTime( tree );
	    return( false );
	}
    }

    /**
     * Get a list of children for the current node applying correct activity
     * semantics.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>TreeNode[]</code> list of active children nodes
     */
    public TreeNode[] getChildren( Tree tree, TreeNode node ) {

	// check if this node has already been seen
	if( ! markSeen( node ) ) {
	    // get all children on active links for the current time
	    return( node.getActiveChildren( eval, time ) );
	} else
	    return( new TreeNode[0] );
    }

    /**
     * Do something with the tree before continuing with next period.
     *
     * @exception <code>VisitException</code> if processing fails
     */
    protected abstract void process( Tree tree ) throws VisitException;
}
