/* 
     File : UpVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A standrard interface for interacting with visitors that
            walk the tree from receivers to source.
             
  $Id: UpVisitor.java,v 1.4 2001/08/22 17:45:07 robertc Exp $
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.util.VisitException;


/**
 * A standrard interface for interacting with visitors that
 * walk the tree from receivers to source.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public interface UpVisitor extends Visitor {

    /**
     * Whether to walk up the tree.
     *
     * @return <code>boolean</code> whether to walk up the tree
     */
    public boolean walkUp();

    /**
     * Check whether the next receiver should start a visiting path.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>Receiver</code> next receiver
     * @return <code>boolean</code> whether to visit next receiver
     */
    public boolean nextReceiver( Tree tree, Receiver recv );

    /**
     * Get a list of active receivers.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>Enumeration</code> list of receivers to visit
     */
    public Enumeration getReceivers( Tree tree );

    /**
     * Visit the current node prior to visiting parent.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitUp( Tree tree, TreeNode node ) throws VisitException;

    /**
     * Visit the current node after visiting parent.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Object</code> list of return values passed back along the path
     * @return <code>Vector</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitUp( Tree tree, TreeNode node, Vector list ) throws VisitException;

    /**
     * Get a list of parents for the current node applying correct activity
     * semantics.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>TreeNode[]</code> array of parent nodes of currently visited node
     */
    public TreeNode[] getParents( Tree tree, TreeNode node );
}
